// ===== MAIN JAVASCRIPT FILE =====

// Global variables
let currentTestimonial = 0;
let testimonialInterval;
const testimonials = [];

// ===== UTILITY FUNCTIONS =====
function debounce(func, wait) {
    let timeout;
    return function executedFunction(...args) {
        const later = () => {
            clearTimeout(timeout);
            func(...args);
        };
        clearTimeout(timeout);
        timeout = setTimeout(later, wait);
    };
}

function throttle(func, limit) {
    let inThrottle;
    return function() {
        const args = arguments;
        const context = this;
        if (!inThrottle) {
            func.apply(context, args);
            inThrottle = true;
            setTimeout(() => inThrottle = false, limit);
        }
    }
}

// ===== INCLUDE LOADER =====
function loadIncludes() {
    // Load header
    const headerPlaceholder = document.getElementById('header-placeholder');
    if (headerPlaceholder) {
        fetch('../../includes/header.html')
            .then(response => response.text())
            .then(data => {
                headerPlaceholder.innerHTML = data;
                initializeHeader();
            })
            .catch(error => console.error('Error loading header:', error));
    }

    // Load footer
    const footerPlaceholder = document.getElementById('footer-placeholder');
    if (footerPlaceholder) {
        fetch('../../includes/footer.html')
            .then(response => response.text())
            .then(data => {
                footerPlaceholder.innerHTML = data;
                initializeFooter();
            })
            .catch(error => console.error('Error loading footer:', error));
    }

    // Load testimonials
    const testimonialsPlaceholder = document.getElementById('testimonials-placeholder');
    if (testimonialsPlaceholder) {
        fetch('../../includes/testimonials.html')
            .then(response => response.text())
            .then(data => {
                testimonialsPlaceholder.innerHTML = data;
                initializeTestimonials();
            })
            .catch(error => console.error('Error loading testimonials:', error));
    }
}

// ===== HEADER FUNCTIONALITY =====
function initializeHeader() {
    const header = document.getElementById('header');
    const mobileMenuToggle = document.getElementById('mobile-menu-toggle');
    const mobileMenuOverlay = document.getElementById('mobile-menu-overlay');
    const nav = document.getElementById('nav');
    const langToggle = document.getElementById('lang-toggle');

    // Scroll effect
    const handleScroll = throttle(() => {
        if (window.scrollY > 50) {
            header.classList.add('scrolled');
        } else {
            header.classList.remove('scrolled');
        }
    }, 100);

    window.addEventListener('scroll', handleScroll);

    // Mobile menu toggle
    if (mobileMenuToggle && nav && mobileMenuOverlay) {
        mobileMenuToggle.addEventListener('click', () => {
            mobileMenuToggle.classList.toggle('active');
            nav.classList.toggle('active');
            mobileMenuOverlay.classList.toggle('active');
            document.body.style.overflow = nav.classList.contains('active') ? 'hidden' : '';
        });

        mobileMenuOverlay.addEventListener('click', () => {
            mobileMenuToggle.classList.remove('active');
            nav.classList.remove('active');
            mobileMenuOverlay.classList.remove('active');
            document.body.style.overflow = '';
        });
    }

    // Language selector
    if (langToggle) {
        const langMenu = langToggle.nextElementSibling;
        
        langToggle.addEventListener('click', (e) => {
            e.stopPropagation();
            langMenu.style.opacity = langMenu.style.opacity === '1' ? '0' : '1';
            langMenu.style.visibility = langMenu.style.visibility === 'visible' ? 'hidden' : 'visible';
        });

        document.addEventListener('click', () => {
            langMenu.style.opacity = '0';
            langMenu.style.visibility = 'hidden';
        });
    }

    // Active nav link
    const currentPath = window.location.pathname;
    const navLinks = document.querySelectorAll('.nav-link');
    
    navLinks.forEach(link => {
        if (link.getAttribute('href') === currentPath || 
            (currentPath.includes(link.getAttribute('href')) && link.getAttribute('href') !== '/')) {
            link.classList.add('active');
        }
    });
}

// ===== FOOTER FUNCTIONALITY =====
function initializeFooter() {
    // Newsletter form
    const newsletterForm = document.getElementById('newsletter-form');
    if (newsletterForm) {
        newsletterForm.addEventListener('submit', (e) => {
            e.preventDefault();
            const email = newsletterForm.querySelector('input[type="email"]').value;
            
            // Simulate API call
            const button = newsletterForm.querySelector('button');
            const originalHTML = button.innerHTML;
            
            button.innerHTML = '<i class="fas fa-spinner fa-spin"></i>';
            button.disabled = true;
            
            setTimeout(() => {
                button.innerHTML = '<i class="fas fa-check"></i>';
                setTimeout(() => {
                    button.innerHTML = originalHTML;
                    button.disabled = false;
                    newsletterForm.reset();
                    showNotification('Thank you for subscribing!', 'success');
                }, 1000);
            }, 2000);
        });
    }

    // Back to top button
    const backToTop = document.getElementById('back-to-top');
    if (backToTop) {
        const handleScroll = throttle(() => {
            if (window.scrollY > 300) {
                backToTop.classList.add('visible');
            } else {
                backToTop.classList.remove('visible');
            }
        }, 100);

        window.addEventListener('scroll', handleScroll);

        backToTop.addEventListener('click', () => {
            window.scrollTo({
                top: 0,
                behavior: 'smooth'
            });
        });
    }

    // Cookie consent
    initializeCookieConsent();
}

// ===== COOKIE CONSENT =====
function initializeCookieConsent() {
    const cookieConsent = document.getElementById('cookie-consent');
    const cookieAccept = document.getElementById('cookie-accept');
    const cookieSettings = document.getElementById('cookie-settings');

    if (!cookieConsent) return;

    // Check if user has already consented
    if (!localStorage.getItem('cookieConsent')) {
        setTimeout(() => {
            cookieConsent.classList.add('visible');
        }, 2000);
    }

    if (cookieAccept) {
        cookieAccept.addEventListener('click', () => {
            localStorage.setItem('cookieConsent', 'accepted');
            cookieConsent.classList.remove('visible');
        });
    }

    if (cookieSettings) {
        cookieSettings.addEventListener('click', () => {
            // Open cookie settings modal (implement as needed)
            console.log('Cookie settings clicked');
        });
    }
}

// ===== TESTIMONIALS SLIDER =====
function initializeTestimonials() {
    const slider = document.getElementById('testimonials-slider');
    const track = slider?.querySelector('.testimonials-track');
    const cards = track?.querySelectorAll('.testimonial-card');
    const prevBtn = document.getElementById('testimonials-prev');
    const nextBtn = document.getElementById('testimonials-next');
    const dotsContainer = document.getElementById('testimonials-dots');

    if (!slider || !track || !cards.length) return;

    // Create dots
    if (dotsContainer) {
        cards.forEach((_, index) => {
            const dot = document.createElement('button');
            dot.classList.add('dot');
            dot.setAttribute('aria-label', `Go to testimonial ${index + 1}`);
            if (index === 0) dot.classList.add('active');
            
            dot.addEventListener('click', () => goToTestimonial(index));
            dotsContainer.appendChild(dot);
        });
    }

    function updateSlider() {
        const translateX = -currentTestimonial * 100;
        track.style.transform = `translateX(${translateX}%)`;
        
        // Update dots
        const dots = dotsContainer?.querySelectorAll('.dot');
        dots?.forEach((dot, index) => {
            dot.classList.toggle('active', index === currentTestimonial);
        });
    }

    function goToTestimonial(index) {
        currentTestimonial = index;
        updateSlider();
        resetAutoplay();
    }

    function nextTestimonial() {
        currentTestimonial = (currentTestimonial + 1) % cards.length;
        updateSlider();
    }

    function prevTestimonial() {
        currentTestimonial = (currentTestimonial - 1 + cards.length) % cards.length;
        updateSlider();
    }

    function startAutoplay() {
        testimonialInterval = setInterval(nextTestimonial, 5000);
    }

    function resetAutoplay() {
        clearInterval(testimonialInterval);
        startAutoplay();
    }

    // Event listeners
    if (nextBtn) nextBtn.addEventListener('click', () => {
        nextTestimonial();
        resetAutoplay();
    });

    if (prevBtn) prevBtn.addEventListener('click', () => {
        prevTestimonial();
        resetAutoplay();
    });

    // Touch/swipe support
    let startX = 0;
    let currentX = 0;
    let isDragging = false;

    track.addEventListener('touchstart', (e) => {
        startX = e.touches[0].clientX;
        isDragging = true;
        clearInterval(testimonialInterval);
    });

    track.addEventListener('touchmove', (e) => {
        if (!isDragging) return;
        currentX = e.touches[0].clientX;
    });

    track.addEventListener('touchend', () => {
        if (!isDragging) return;
        isDragging = false;
        
        const diffX = startX - currentX;
        const threshold = 50;
        
        if (Math.abs(diffX) > threshold) {
            if (diffX > 0) {
                nextTestimonial();
            } else {
                prevTestimonial();
            }
        }
        
        resetAutoplay();
    });

    // Pause on hover
    slider.addEventListener('mouseenter', () => clearInterval(testimonialInterval));
    slider.addEventListener('mouseleave', startAutoplay);

    // Start autoplay
    startAutoplay();
}

// ===== PRICING TOGGLE =====
function initPricingToggle() {
    const toggle = document.querySelector('.toggle-switch input');
    const monthlyPrices = document.querySelectorAll('.monthly-price');
    const yearlyPrices = document.querySelectorAll('.yearly-price');

    if (!toggle) return;

    toggle.addEventListener('change', () => {
        const isYearly = toggle.checked;
        
        monthlyPrices.forEach(price => {
            price.style.display = isYearly ? 'none' : 'inline';
        });
        
        yearlyPrices.forEach(price => {
            price.style.display = isYearly ? 'inline' : 'none';
        });
    });
}

// ===== COUNTER ANIMATION =====
function initCounters() {
    const counters = document.querySelectorAll('.stat-number');
    
    const animateCounter = (counter) => {
        const target = parseInt(counter.getAttribute('data-count'));
        const duration = 2000;
        const step = target / (duration / 16);
        let current = 0;
        
        const updateCounter = () => {
            current += step;
            if (current < target) {
                counter.textContent = Math.floor(current);
                requestAnimationFrame(updateCounter);
            } else {
                counter.textContent = target;
            }
        };
        
        updateCounter();
    };

    // Intersection Observer for counters
    const counterObserver = new IntersectionObserver((entries) => {
        entries.forEach(entry => {
            if (entry.isIntersecting) {
                animateCounter(entry.target);
                counterObserver.unobserve(entry.target);
            }
        });
    }, { threshold: 0.5 });

    counters.forEach(counter => {
        counterObserver.observe(counter);
    });
}

// ===== FORM HANDLING =====
function initForms() {
    const forms = document.querySelectorAll('form[data-form]');
    
    forms.forEach(form => {
        form.addEventListener('submit', async (e) => {
            e.preventDefault();
            
            const formData = new FormData(form);
            const data = Object.fromEntries(formData);
            const submitBtn = form.querySelector('button[type="submit"]');
            const originalText = submitBtn.textContent;
            
            // Show loading state
            submitBtn.disabled = true;
            submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Processing...';
            
            try {
                // Simulate API call
                await new Promise(resolve => setTimeout(resolve, 2000));
                
                // Success
                showNotification('Form submitted successfully!', 'success');
                form.reset();
                
            } catch (error) {
                // Error
                showNotification('Something went wrong. Please try again.', 'error');
                console.error('Form submission error:', error);
                
            } finally {
                // Reset button
                submitBtn.disabled = false;
                submitBtn.textContent = originalText;
            }
        });
    });
}

    // ===== NOTIFICATION SYSTEM =====
function showNotification(message, type = 'info', duration = 5000) {
    const notification = document.createElement('div');
    notification.className = `notification notification-${type}`;
    notification.innerHTML = `
        <div class="notification-content">
            <i class="fas fa-${getNotificationIcon(type)}"></i>
            <span>${message}</span>
            <button class="notification-close" aria-label="Close notification">
                <i class="fas fa-times"></i>
            </button>
        </div>
    `;
    
    // Add styles
    notification.style.cssText = `
        position: fixed;
        top: 20px;
        right: 20px;
        background: ${getNotificationColor(type)};
        color: white;
        padding: 16px 20px;
        border-radius: 8px;
        box-shadow: 0 4px 12px rgba(0,0,0,0.15);
        z-index: 1080;
        transform: translateX(100%);
        transition: transform 0.3s ease;
        max-width: 400px;
    `;
    
    notification.querySelector('.notification-content').style.cssText = `
        display: flex;
        align-items: center;
        gap: 12px;
    `;
    
    notification.querySelector('.notification-close').style.cssText = `
        background: none;
        border: none;
        color: white;
        cursor: pointer;
        padding: 4px;
        margin-left: auto;
    `;
    
    document.body.appendChild(notification);
    
    // Animate in
    setTimeout(() => {
        notification.style.transform = 'translateX(0)';
    }, 100);
    
    // Close functionality
    const closeBtn = notification.querySelector('.notification-close');
    const closeNotification = () => {
        notification.style.transform = 'translateX(100%)';
        setTimeout(() => {
            if (notification.parentNode) {
                notification.parentNode.removeChild(notification);
            }
        }, 300);
    };
    
    closeBtn.addEventListener('click', closeNotification);
    
    // Auto close
    if (duration > 0) {
        setTimeout(closeNotification, duration);
    }
}

function getNotificationIcon(type) {
    const icons = {
        success: 'check-circle',
        error: 'exclamation-circle',
        warning: 'exclamation-triangle',
        info: 'info-circle'
    };
    return icons[type] || icons.info;
}

function getNotificationColor(type) {
    const colors = {
        success: '#10b981',
        error: '#ef4444',
        warning: '#f59e0b',
        info: '#3b82f6'
    };
    return colors[type] || colors.info;
}

// ===== LAZY LOADING =====
function initLazyLoading() {
    const images = document.querySelectorAll('img[loading="lazy"]');
    
    if ('IntersectionObserver' in window) {
        const imageObserver = new IntersectionObserver((entries) => {
            entries.forEach(entry => {
                if (entry.isIntersecting) {
                    const img = entry.target;
                    img.src = img.dataset.src || img.src;
                    img.classList.remove('lazy');
                    imageObserver.unobserve(img);
                }
            });
        });
        
        images.forEach(img => {
            img.classList.add('lazy');
            imageObserver.observe(img);
        });
    }
}

// ===== SMOOTH SCROLLING =====
function initSmoothScrolling() {
    const links = document.querySelectorAll('a[href^="#"]');
    
    links.forEach(link => {
        link.addEventListener('click', (e) => {
            const href = link.getAttribute('href');
            if (href === '#') return;
            
            const target = document.querySelector(href);
            if (target) {
                e.preventDefault();
                
                const headerHeight = document.getElementById('header')?.offsetHeight || 0;
                const targetPosition = target.offsetTop - headerHeight - 20;
                
                window.scrollTo({
                    top: targetPosition,
                    behavior: 'smooth'
                });
            }
        });
    });
}

// ===== SEARCH FUNCTIONALITY =====
function initSearch() {
    const searchInput = document.getElementById('search-input');
    const searchResults = document.getElementById('search-results');
    
    if (!searchInput || !searchResults) return;
    
    const performSearch = debounce(async (query) => {
        if (query.length < 2) {
            searchResults.innerHTML = '';
            searchResults.style.display = 'none';
            return;
        }
        
        try {
            // Simulate search API call
            const results = await mockSearch(query);
            displaySearchResults(results);
        } catch (error) {
            console.error('Search error:', error);
        }
    }, 300);
    
    searchInput.addEventListener('input', (e) => {
        performSearch(e.target.value.trim());
    });
    
    // Close search results when clicking outside
    document.addEventListener('click', (e) => {
        if (!searchInput.contains(e.target) && !searchResults.contains(e.target)) {
            searchResults.style.display = 'none';
        }
    });
}

async function mockSearch(query) {
    // Mock search data
    const mockData = [
        { title: 'Shared Hosting', url: '/products/shared-hosting.html', type: 'Product' },
        { title: 'VPS Hosting', url: '/products/vps-hosting.html', type: 'Product' },
        { title: 'SEO Services', url: '/products/seo-services.html', type: 'Service' },
        { title: 'Content Creation', url: '/products/content-creation.html', type: 'Service' },
        { title: 'About Us', url: '/locales/en/about.html', type: 'Page' },
        { title: 'Contact', url: '/locales/en/contact.html', type: 'Page' },
        { title: 'Support Center', url: '/support/', type: 'Page' },
        { title: 'Pricing', url: '/locales/en/pricing.html', type: 'Page' }
    ];
    
    return new Promise(resolve => {
        setTimeout(() => {
            const results = mockData.filter(item => 
                item.title.toLowerCase().includes(query.toLowerCase())
            );
            resolve(results);
        }, 200);
    });
}

function displaySearchResults(results) {
    const searchResults = document.getElementById('search-results');
    
    if (results.length === 0) {
        searchResults.innerHTML = '<div class="search-no-results">No results found</div>';
    } else {
        searchResults.innerHTML = results.map(result => `
            <a href="${result.url}" class="search-result-item">
                <div class="search-result-title">${result.title}</div>
                <div class="search-result-type">${result.type}</div>
            </a>
        `).join('');
    }
    
    searchResults.style.display = 'block';
}

// ===== PERFORMANCE MONITORING =====
function initPerformanceMonitoring() {
    // Monitor page load performance
    window.addEventListener('load', () => {
        if ('performance' in window) {
            const perfData = performance.getEntriesByType('navigation')[0];
            const loadTime = perfData.loadEventEnd - perfData.loadEventStart;
            
            console.log(`Page load time: ${loadTime}ms`);
            
            // Send to analytics (implement as needed)
            if (loadTime > 3000) {
                console.warn('Slow page load detected');
            }
        }
    });
    
    // Monitor Core Web Vitals
    if ('web-vital' in window) {
        // This would require the web-vitals library
        // getCLS(console.log);
        // getFID(console.log);
        // getLCP(console.log);
    }
}

// ===== ERROR HANDLING =====
function initErrorHandling() {
    window.addEventListener('error', (e) => {
        console.error('JavaScript Error:', e.error);
        
        // Send error to monitoring service (implement as needed)
        // sendErrorToService({
        //     message: e.message,
        //     filename: e.filename,
        //     lineno: e.lineno,
        //     colno: e.colno,
        //     stack: e.error?.stack
        // });
    });
    
    window.addEventListener('unhandledrejection', (e) => {
        console.error('Unhandled Promise Rejection:', e.reason);
        
        // Send error to monitoring service
        // sendErrorToService({
        //     type: 'unhandledrejection',
        //     reason: e.reason
        // });
    });
}

// ===== ANALYTICS =====
function initAnalytics() {
    // Google Analytics 4 (replace with your measurement ID)
    if (typeof gtag !== 'undefined') {
        // Track page views
        gtag('config', 'GA_MEASUREMENT_ID', {
            page_title: document.title,
            page_location: window.location.href
        });
        
        // Track custom events
        document.addEventListener('click', (e) => {
            const target = e.target.closest('[data-track]');
            if (target) {
                const action = target.dataset.track;
                const category = target.dataset.trackCategory || 'engagement';
                const label = target.dataset.trackLabel || target.textContent.trim();
                
                gtag('event', action, {
                    event_category: category,
                    event_label: label
                });
            }
        });
    }
    
    // Track form submissions
    document.addEventListener('submit', (e) => {
        const form = e.target;
        if (form.dataset.trackForm) {
            if (typeof gtag !== 'undefined') {
                gtag('event', 'form_submit', {
                    event_category: 'forms',
                    event_label: form.dataset.trackForm
                });
            }
        }
    });
}

// ===== ACCESSIBILITY ENHANCEMENTS =====
function initAccessibility() {
    // Skip link functionality
    const skipLink = document.querySelector('.skip-link');
    if (skipLink) {
        skipLink.addEventListener('click', (e) => {
            e.preventDefault();
            const target = document.querySelector(skipLink.getAttribute('href'));
            if (target) {
                target.focus();
                target.scrollIntoView();
            }
        });
    }
    
    // Keyboard navigation for dropdowns
    const dropdowns = document.querySelectorAll('.dropdown');
    dropdowns.forEach(dropdown => {
        const trigger = dropdown.querySelector('.nav-link');
        const menu = dropdown.querySelector('.dropdown-menu');
        
        if (trigger && menu) {
            trigger.addEventListener('keydown', (e) => {
                if (e.key === 'Enter' || e.key === ' ') {
                    e.preventDefault();
                    menu.style.opacity = menu.style.opacity === '1' ? '0' : '1';
                    menu.style.visibility = menu.style.visibility === 'visible' ? 'hidden' : 'visible';
                }
            });
        }
    });
    
    // Announce dynamic content changes to screen readers
    function announceToScreenReader(message) {
        const announcement = document.createElement('div');
        announcement.setAttribute('aria-live', 'polite');
        announcement.setAttribute('aria-atomic', 'true');
        announcement.className = 'sr-only';
        announcement.textContent = message;
        
        document.body.appendChild(announcement);
        
        setTimeout(() => {
            document.body.removeChild(announcement);
        }, 1000);
    }
    
    // Make announcements available globally
    window.announceToScreenReader = announceToScreenReader;
}

// ===== THEME MANAGEMENT =====
function initThemeManagement() {
    const themeToggle = document.getElementById('theme-toggle');
    const prefersDark = window.matchMedia('(prefers-color-scheme: dark)');
    
    // Get saved theme or default to system preference
    let currentTheme = localStorage.getItem('theme') || 
                      (prefersDark.matches ? 'dark' : 'light');
    
    function setTheme(theme) {
        document.documentElement.setAttribute('data-theme', theme);
        localStorage.setItem('theme', theme);
        currentTheme = theme;
        
        if (themeToggle) {
            themeToggle.setAttribute('aria-label', 
                theme === 'dark' ? 'Switch to light mode' : 'Switch to dark mode'
            );
        }
    }
    
    // Initialize theme
    setTheme(currentTheme);
    
    // Theme toggle functionality
    if (themeToggle) {
        themeToggle.addEventListener('click', () => {
            const newTheme = currentTheme === 'dark' ? 'light' : 'dark';
            setTheme(newTheme);
        });
    }
    
    // Listen for system theme changes
    prefersDark.addEventListener('change', (e) => {
        if (!localStorage.getItem('theme')) {
            setTheme(e.matches ? 'dark' : 'light');
        }
    });
}

// ===== SERVICE WORKER =====
function initServiceWorker() {
    if ('serviceWorker' in navigator) {
        window.addEventListener('load', () => {
            navigator.serviceWorker.register('/sw.js')
                .then(registration => {
                    console.log('SW registered: ', registration);
                })
                .catch(registrationError => {
                    console.log('SW registration failed: ', registrationError);
                });
        });
    }
}

// ===== MAIN INITIALIZATION =====
document.addEventListener('DOMContentLoaded', () => {
    // Load includes first
    loadIncludes();
    
    // Initialize other functionality
    initPricingToggle();
    initCounters();
    initForms();
    initLazyLoading();
    initSmoothScrolling();
    initSearch();
    initPerformanceMonitoring();
    initErrorHandling();
    initAnalytics();
    initAccessibility();
    initThemeManagement();
    initServiceWorker();
    
    // Add loading complete class
    document.body.classList.add('loaded');
    
    // Animate elements on scroll
    const animateOnScroll = new IntersectionObserver((entries) => {
        entries.forEach(entry => {
            if (entry.isIntersecting) {
                entry.target.classList.add('fade-in');
                animateOnScroll.unobserve(entry.target);
            }
        });
    }, { threshold: 0.1 });
    
    // Observe elements for animation
    document.querySelectorAll('.service-card, .feature-item, .pricing-card, .stat-item')
        .forEach(el => animateOnScroll.observe(el));
});

// ===== WINDOW LOAD EVENT =====
window.addEventListener('load', () => {
    // Hide loading spinner if present
    const loader = document.getElementById('page-loader');
    if (loader) {
        loader.style.opacity = '0';
        setTimeout(() => {
            loader.style.display = 'none';
        }, 300);
    }
    
    // Initialize components that need full page load
    initCounters();
});

// ===== RESIZE HANDLER =====
window.addEventListener('resize', debounce(() => {
    // Handle responsive adjustments
    const mobileBreakpoint = 768;
    const isMobile = window.innerWidth < mobileBreakpoint;
    
    // Update mobile-specific functionality
    document.body.classList.toggle('mobile', isMobile);
    
    // Reset mobile menu if switching to desktop
    if (!isMobile) {
        const nav = document.getElementById('nav');
        const mobileMenuToggle = document.getElementById('mobile-menu-toggle');
        const mobileMenuOverlay = document.getElementById('mobile-menu-overlay');
        
        if (nav) nav.classList.remove('active');
        if (mobileMenuToggle) mobileMenuToggle.classList.remove('active');
        if (mobileMenuOverlay) mobileMenuOverlay.classList.remove('active');
        document.body.style.overflow = '';
    }
}, 250));

// ===== EXPORT FUNCTIONS FOR EXTERNAL USE =====
window.SmarterswebJS = {
    showNotification,
    announceToScreenReader: () => {}, // Will be set in initAccessibility
    loadIncludes,
    initTestimonials,
    initPricingToggle
};