<?php
session_start();

define('APP_PATH', realpath(__DIR__ . '/../app'));
require_once APP_PATH . '/Config/Config.php';
require_once APP_PATH . '/Config/Database.php';
require_once APP_PATH . '/Models/User.php';
require_once APP_PATH . '/Models/Form.php';

if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit();
}

$database = new Database();
$db = $database->getConnection();

$user = new User($db);
$form = new Form($db);

$user->getById($_SESSION['user_id']);

$action = $_GET['action'] ?? 'list';
$form_id = $_GET['id'] ?? null;

$message = '';
$error = '';

// Helper to safely get POST values
function getPostVal(string $key, $default = null) {
    return $_POST[$key] ?? $default;
}

// Handle form creation
if ($action === 'create' && $_SERVER['REQUEST_METHOD'] === 'POST') {
    $form->user_id = $_SESSION['user_id'];
    $form->form_name = trim(getPostVal('form_name', ''));
    
    $fieldsJson = getPostVal('fields', '[]');
    $fieldsArray = json_decode($fieldsJson, true);
    if (!is_array($fieldsArray)) {
        $fieldsArray = [];
    }
    
    $form->form_config = [
        'redirect_url' => trim(getPostVal('redirect_url', '')),
        'email_notifications' => isset($_POST['email_notifications']),
        'notification_email' => trim(getPostVal('notification_email', $_SESSION['email'] ?? '')),
        'spam_protection' => isset($_POST['spam_protection']),
        'fields' => $fieldsArray
    ];

    if ($form->create()) {
        $message = "Form created successfully!";
        $action = 'edit';
        $form_id = $form->id;
    } else {
        $error = "Failed to create form.";
    }
}

// Handle form editing
if ($action === 'edit' && $form_id) {
    $form_query = "SELECT * FROM forms WHERE id = :id AND user_id = :user_id LIMIT 1";
    $form_stmt = $db->prepare($form_query);
    $form_stmt->bindParam(':id', $form_id, PDO::PARAM_INT);
    $form_stmt->bindParam(':user_id', $_SESSION['user_id'], PDO::PARAM_INT);
    $form_stmt->execute();
    $current_form = $form_stmt->fetch(PDO::FETCH_ASSOC);

    if (!$current_form) {
        header('Location: forms.php');
        exit();
    }

    if ($_SERVER['REQUEST_METHOD'] === 'POST') {
        $fieldsJson = getPostVal('fields', '[]');
        $fieldsArray = json_decode($fieldsJson, true);
        if (!is_array($fieldsArray)) {
            $fieldsArray = [];
        }

        $update_query = "UPDATE forms SET 
            form_name = :form_name,
            form_config = :form_config,
            updated_at = CURRENT_TIMESTAMP
            WHERE id = :id AND user_id = :user_id";
        
        $update_stmt = $db->prepare($update_query);

        $form_config = [
            'redirect_url' => trim(getPostVal('redirect_url', '')),
            'email_notifications' => isset($_POST['email_notifications']),
            'notification_email' => trim(getPostVal('notification_email', $_SESSION['email'] ?? '')),
            'spam_protection' => isset($_POST['spam_protection']),
            'fields' => $fieldsArray
        ];

        $update_stmt->bindParam(':form_name', $_POST['form_name']);
        $update_stmt->bindParam(':form_config', json_encode($form_config));
        $update_stmt->bindParam(':id', $form_id, PDO::PARAM_INT);
        $update_stmt->bindParam(':user_id', $_SESSION['user_id'], PDO::PARAM_INT);

        if ($update_stmt->execute()) {
            $message = "Form updated successfully!";
            // Refresh current_form data
            $form_stmt->execute();
            $current_form = $form_stmt->fetch(PDO::FETCH_ASSOC);
        } else {
            $error = "Failed to update form.";
        }
    }
}

// Get all user forms for listing
$user_forms = $form->getByUserId($_SESSION['user_id']);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8" />
    <meta name="viewport" content="width=device-width, initial-scale=1" />
    <title>Forms - SmartForm</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet" />
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet" />
    <link href="https://cdn.jsdelivr.net/npm/sortablejs@1.15.0/Sortable.min.css" rel="stylesheet" />
    <link href="assets/css/style.css" rel="stylesheet" />
</head>
<body>
<nav class="navbar navbar-expand-lg navbar-dark bg-primary">
    <div class="container-fluid">
        <a class="navbar-brand fw-bold" href="index.php"><i class="fas fa-bolt"></i> SmartForm</a>
        <div class="navbar-nav ms-auto">
            <div class="nav-item dropdown">
                <a class="nav-link dropdown-toggle" href="#" id="navbarDropdown" role="button" data-bs-toggle="dropdown">
                    <i class="fas fa-user-circle me-1"></i> <?php echo htmlspecialchars($user->username); ?>
                </a>
                <ul class="dropdown-menu">
                    <li><a class="dropdown-item" href="profile.php"><i class="fas fa-user me-2"></i>Profile</a></li>
                    <li><a class="dropdown-item" href="billing.php"><i class="fas fa-credit-card me-2"></i>Billing</a></li>
                    <li><hr class="dropdown-divider" /></li>
                    <li><a class="dropdown-item" href="logout.php"><i class="fas fa-sign-out-alt me-2"></i>Logout</a></li>
                </ul>
            </div>
        </div>
    </div>
</nav>

<div class="container-fluid">
    <div class="row">
        <!-- Sidebar omitted for brevity, unchanged -->

        <div class="col-md-9 col-lg-10">
            <div class="p-4">
                <?php if ($action === 'list'): ?>
                <!-- Forms List -->
                <!-- ... Your forms listing HTML here (unchanged) ... -->

                <?php elseif ($action === 'create' || $action === 'edit'): ?>
                <form method="POST" id="formBuilderForm" action="">
                    <!-- Form Settings & Builder HTML -->

                    <!-- Hidden field with JSON fields config -->
                    <input type="hidden" name="fields" id="fieldsConfig" value="<?php
                        echo isset($current_form)
                            ? htmlspecialchars(json_encode(json_decode($current_form['form_config'], true)['fields'] ?? []))
                            : '[]';
                    ?>">
                </form>
                <?php endif; ?>
            </div>
        </div>
    </div>
</div>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/sortablejs@1.15.0/Sortable.min.js"></script>
<script src="assets/js/main.js"></script>

<?php if ($action === 'create' || $action === 'edit'): ?>
<script>
class FormBuilderApp {
    constructor() {
        this.fields = <?php
            echo isset($current_form)
                ? json_encode(json_decode($current_form['form_config'], true)['fields'] ?? [])
                : '[]';
        ?>;
        this.init();
    }

    init() {
        this.setupFieldPalette();
        this.setupFormPreview();
        this.setupEventListeners();
        this.updatePreview();
    }

    setupFieldPalette() {
        const palette = document.getElementById('fieldPalette');
        const fieldTypes = [
            { type: 'text', label: 'Text Input', icon: 'fa-font' },
            { type: 'email', label: 'Email', icon: 'fa-envelope' },
            { type: 'tel', label: 'Phone', icon: 'fa-phone' },
            { type: 'textarea', label: 'Textarea', icon: 'fa-align-left' },
            { type: 'select', label: 'Select Dropdown', icon: 'fa-list' },
            { type: 'checkbox', label: 'Checkbox', icon: 'fa-check-square' },
            { type: 'radio', label: 'Radio Buttons', icon: 'fa-dot-circle' },
            { type: 'file', label: 'File Upload', icon: 'fa-upload' },
            { type: 'number', label: 'Number', icon: 'fa-hashtag' },
            { type: 'date', label: 'Date', icon: 'fa-calendar' }
        ];

        fieldTypes.forEach(field => {
            const fieldElement = document.createElement('div');
            fieldElement.className = 'field-item';
            fieldElement.innerHTML = `<i class="fas ${field.icon} me-2"></i>${field.label}`;
            fieldElement.addEventListener('click', () => this.addField(field));
            palette.appendChild(fieldElement);
        });
    }

    setupFormPreview() {
        const preview = document.getElementById('formPreview');
        Sortable.create(preview, {
            group: 'form-fields',
            animation: 150,
            ghostClass: 'sortable-ghost',
            chosenClass: 'sortable-chosen',
            onUpdate: () => this.updateFieldsOrder()
        });
    }

    setupEventListeners() {
        document.getElementById('previewBtn').addEventListener('click', () => this.showPreview());
        document.getElementById('codeBtn').addEventListener('click', () => this.showCode());
        document.getElementById('clearForm').addEventListener('click', () => this.clearForm());
        document.getElementById('formBuilderForm').addEventListener('submit', e => this.saveForm(e));

        // Copy code button
        document.querySelectorAll('[data-copy-target]').forEach(btn => {
            btn.addEventListener('click', e => {
                const targetSelector = btn.getAttribute('data-copy-target');
                const target = document.querySelector(targetSelector);
                if (!target) return;
                target.select();
                document.execCommand('copy');
                btn.innerHTML = '<i class="fas fa-check me-2"></i>Copied!';
                setTimeout(() => {
                    btn.innerHTML = '<i class="fas fa-copy me-2"></i>Copy Code';
                }, 2000);
            });
        });
    }

    addField(fieldType) {
        const id = 'field_' + Date.now();
        const field = {
            id,
            type: fieldType.type,
            label: fieldType.label,
            name: fieldType.type + '_' + Date.now(),
            required: false,
            placeholder: '',
            options: ['select', 'radio'].includes(fieldType.type) ? ['Option 1', 'Option 2'] : []
        };
        this.fields.push(field);
        this.updatePreview();
    }

    removeField(fieldId) {
        this.fields = this.fields.filter(f => f.id !== fieldId);
        this.updatePreview();
    }

    editField(fieldId) {
        const field = this.fields.find(f => f.id === fieldId);
        if (!field) return;
        const newLabel = prompt('Enter field label:', field.label);
        if (newLabel) {
            field.label = newLabel;
            this.updatePreview();
        }
    }

    updatePreview() {
        const preview = document.getElementById('formPreview');

        if (this.fields.length === 0) {
            preview.innerHTML = `
                <div class="text-center text-muted py-5">
                    <i class="fas fa-mouse-pointer fa-2x mb-3"></i>
                    <p>Drag fields from the left panel to build your form</p>
                </div>`;
            document.getElementById('fieldsConfig').value = '[]';
            return;
        }

        let html = '';
        this.fields.forEach(field => {
            html += this.generateFieldHTML(field);
        });

        preview.innerHTML = html;

        // Edit and remove button handlers
        preview.querySelectorAll('.edit-field').forEach(btn => {
            btn.addEventListener('click', e => {
                e.preventDefault();
                this.editField(btn.getAttribute('data-field-id'));
            });
        });
        preview.querySelectorAll('.remove-field').forEach(btn => {
            btn.addEventListener('click', e => {
                e.preventDefault();
                this.removeField(btn.getAttribute('data-field-id'));
            });
        });

        document.getElementById('fieldsConfig').value = JSON.stringify(this.fields);
    }

    generateFieldHTML(field) {
        let html = `
        <div class="mb-3 field-container" data-field-id="${field.id}">
            <div class="d-flex justify-content-between align-items-center mb-2">
                <label class="form-label">${field.label}${field.required ? ' *' : ''}</label>
                <div class="btn-group btn-group-sm">
                    <button type="button" class="btn btn-outline-secondary edit-field" data-field-id="${field.id}" title="Edit Field">
                        <i class="fas fa-edit"></i>
                    </button>
                    <button type="button" class="btn btn-outline-danger remove-field" data-field-id="${field.id}" title="Remove Field">
                        <i class="fas fa-trash"></i>
                    </button>
                </div>
            </div>`;

        switch (field.type) {
            case 'text':
            case 'email':
            case 'tel':
            case 'number':
            case 'date':
                html += `<input type="${field.type}" class="form-control" name="${field.name}" placeholder="${field.placeholder}" ${field.required ? 'required' : ''}>`;
                break;
            case 'textarea':
                html += `<textarea class="form-control" name="${field.name}" rows="4" placeholder="${field.placeholder}" ${field.required ? 'required' : ''}></textarea>`;
                break;
            case 'select':
                html += `<select class="form-select" name="${field.name}" ${field.required ? 'required' : ''}>`;
                html += `<option value="">Choose...</option>`;
                field.options.forEach(option => {
                    html += `<option value="${option}">${option}</option>`;
                });
                html += `</select>`;
                break;
            case 'checkbox':
                html += `
                <div class="form-check">
                    <input class="form-check-input" type="checkbox" name="${field.name}" id="${field.id}" ${field.required ? 'required' : ''}>
                    <label class="form-check-label" for="${field.id}">${field.placeholder || 'Check this box'}</label>
                </div>`;
                break;
            case 'radio':
                field.options.forEach((option, idx) => {
                    html += `
                    <div class="form-check">
                        <input class="form-check-input" type="radio" name="${field.name}" id="${field.id}_${idx}" value="${option}" ${field.required ? 'required' : ''}>
                        <label class="form-check-label" for="${field.id}_${idx}">${option}</label>
                    </div>`;
                });
                break;
            case 'file':
                html += `<input type="file" class="form-control" name="${field.name}" ${field.required ? 'required' : ''}>`;
                break;
        }
        html += '</div>';
        return html;
    }

    showPreview() {
        // Implementation for modal preview (if needed)
    }

    showCode() {
        // Implementation for showing form embed code (if needed)
    }

    clearForm() {
        if (confirm('Are you sure you want to clear all fields?')) {
            this.fields = [];
            this.updatePreview();
        }
    }

    updateFieldsOrder() {
        const containers = document.querySelectorAll('.field-container');
        const newOrder = [];
        containers.forEach(container => {
            const id = container.getAttribute('data-field-id');
            const field = this.fields.find(f => f.id === id);
            if (field) newOrder.push(field);
        });
        this.fields = newOrder;
        document.getElementById('fieldsConfig').value = JSON.stringify(this.fields);
    }

    saveForm(e) {
        // Update hidden input before submit
        document.getElementById('fieldsConfig').value = JSON.stringify(this.fields);
        // Let form submit naturally
    }
}

document.addEventListener('DOMContentLoaded', () => {
    new FormBuilderApp();
});
</script>
<?php endif; ?>
</body>
</html>