<?php
session_start();

require_once '../../app/Config/Config.php';
require_once '../../app/Config/Database.php';
require_once '../../app/Models/Form.php';
require_once '../../app/Models/Submission.php';

$database = new Database();
$db = $database->getConnection();
$form = new Form($db);
$submission = new Submission($db);

// Get form endpoint from URL
$endpoint = $_GET['endpoint'] ?? '';

if (empty($endpoint)) {
    http_response_code(404);
    die('Form not found');
}

// Get form by endpoint
$form_data = $form->getByEndpoint($endpoint);

if (!$form_data || !$form_data['is_active']) {
    http_response_code(404);
    die('Form not found or inactive');
}

$form_config = json_decode($form_data['form_config'], true);
if (!is_array($form_config)) {
    $form_config = [];
}

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Spam protection
    if (!empty($form_config['spam_protection'])) {
        // Honeypot check
        if (!empty($_POST['website'])) {
            http_response_code(400);
            die('Spam detected');
        }

        // Simple rate limiting per IP per form (60 seconds)
        $ip = $_SERVER['REMOTE_ADDR'] ?? '';
        $rate_limit_key = "rate_limit_" . md5($ip . $form_data['id']);
        if (isset($_SESSION[$rate_limit_key]) && $_SESSION[$rate_limit_key] > time() - 60) {
            http_response_code(429);
            die('Rate limit exceeded. Please wait before submitting again.');
        }
        $_SESSION[$rate_limit_key] = time();
    }

    // Process submission data
    $submission_data = [];

    // Copy POST data except system fields
    $filtered_post = $_POST;
    unset($filtered_post['website']); // honeypot

    foreach ($filtered_post as $key => $value) {
        if (is_array($value)) {
            $submission_data[$key] = implode(', ', $value);
        } else {
            $submission_data[$key] = trim($value);
        }
    }

    // Handle file uploads
    if (!empty($_FILES)) {
        $upload_dir = __DIR__ . '/../../uploads/' . $form_data['id'] . '/';
        if (!is_dir($upload_dir)) {
            mkdir($upload_dir, 0755, true);
        }

        foreach ($_FILES as $field_name => $file) {
            if (isset($file['error']) && $file['error'] === UPLOAD_ERR_OK) {
                $file_extension = pathinfo($file['name'], PATHINFO_EXTENSION);
                $safe_filename = uniqid('', true) . '.' . $file_extension;
                $upload_path = $upload_dir . $safe_filename;

                if (move_uploaded_file($file['tmp_name'], $upload_path)) {
                    $submission_data[$field_name] = $safe_filename;
                }
            }
        }
    }

    $submission->form_id = $form_data['id'];
    $submission->submission_data = $submission_data;
    $submission->ip_address = $_SERVER['REMOTE_ADDR'] ?? '';
    $submission->user_agent = $_SERVER['HTTP_USER_AGENT'] ?? '';

    if ($submission->create()) {
        // Email notification
        if (!empty($form_config['email_notifications'])) {
            $to = $form_config['notification_email'] ?? '';
            if (filter_var($to, FILTER_VALIDATE_EMAIL)) {
                $subject = 'New submission for ' . $form_data['form_name'];
                $message = "You have received a new submission for your form: " . $form_data['form_name'] . "\n\n";

                foreach ($submission_data as $key => $value) {
                    $message .= ucfirst(str_replace('_', ' ', $key)) . ": " . $value . "\n";
                }

                $headers = 'From: noreply@' . ($_SERVER['HTTP_HOST'] ?? 'example.com');
                @mail($to, $subject, $message, $headers);
            }
        }

        // Redirect or JSON response
        if (!empty($form_config['redirect_url'])) {
            header('Location: ' . $form_config['redirect_url']);
            exit();
        }

        if (!empty($_SERVER['HTTP_X_REQUESTED_WITH']) &&
            strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) === 'xmlhttprequest') {
            header('Content-Type: application/json');
            echo json_encode(['success' => true, 'message' => 'Form submitted successfully!']);
            exit();
        }

        showSuccessPage($form_data['form_name']);
        exit();
    } else {
        if (!empty($_SERVER['HTTP_X_REQUESTED_WITH']) &&
            strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) === 'xmlhttprequest') {
            header('Content-Type: application/json');
            http_response_code(500);
            echo json_encode(['success' => false, 'message' => 'Failed to submit form. Please try again.']);
            exit();
        }
        showErrorPage();
        exit();
    }
}

// Display form (GET request)
displayForm($form_data, $form_config);

/**
 * Outputs the HTML form based on form data and config
 */
function displayForm($form_data, $form_config) {
    $fields = $form_config['fields'] ?? [];
    ?>
    <!DOCTYPE html>
    <html lang="en">
    <head>
        <meta charset="UTF-8" />
        <meta name="viewport" content="width=device-width, initial-scale=1" />
        <title><?php echo htmlspecialchars($form_data['form_name']); ?></title>
        <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet" />
        <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet" />
        <style>
            body { background: #f8fafc; }
            .form-container { max-width: 600px; margin: 2rem auto; }
            .card { border: none; box-shadow: 0 4px 6px rgba(0,0,0,0.1); }
            /* Hide honeypot off-screen */
            input[name="website"] {
                position: absolute !important;
                left: -9999px !important;
                top: -9999px !important;
                height: 1px !important;
                width: 1px !important;
                opacity: 0 !important;
                pointer-events: none !important;
            }
        </style>
    </head>
    <body>
    <div class="container">
        <div class="form-container">
            <div class="card">
                <div class="card-header bg-primary text-white">
                    <h4 class="mb-0"><?php echo htmlspecialchars($form_data['form_name']); ?></h4>
                </div>
                <div class="card-body">
                    <form method="POST" enctype="multipart/form-data" data-smartform>
                        <?php foreach ($fields as $field): ?>
                            <div class="mb-3">
                                <?php echo generateFieldHTML($field); ?>
                            </div>
                        <?php endforeach; ?>

                        <?php if (!empty($form_config['spam_protection'])): ?>
                            <!-- Honeypot field -->
                            <input type="text" name="website" tabindex="-1" autocomplete="off" />
                        <?php endif; ?>

                        <button type="submit" class="btn btn-primary">
                            <i class="fas fa-paper-plane me-2"></i>Submit
                        </button>
                    </form>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // AJAX form submission
        document.querySelector('form[data-smartform]').addEventListener('submit', async function(e) {
            e.preventDefault();
            const form = e.target;
            const formData = new FormData(form);
            const submitBtn = form.querySelector('button[type="submit"]');
            const originalText = submitBtn.innerHTML;

            submitBtn.disabled = true;
            submitBtn.innerHTML = '<span class="spinner-border spinner-border-sm me-2"></span>Submitting...';

            // Remove previous alerts
            const oldAlerts = form.querySelectorAll('.alert');
            oldAlerts.forEach(alert => alert.remove());

            try {
                const response = await fetch(window.location.href, {
                    method: 'POST',
                    body: formData,
                    headers: {'X-Requested-With': 'XMLHttpRequest'}
                });

                const result = await response.json();

                if (result.success) {
                    form.innerHTML = `
                        <div class="alert alert-success text-center">
                            <i class="fas fa-check-circle fa-2x mb-3"></i>
                            <h5>Thank you!</h5>
                            <p>${result.message}</p>
                        </div>
                    `;
                } else {
                    throw new Error(result.message || 'Submission failed.');
                }
            } catch (err) {
                const alertDiv = document.createElement('div');
                alertDiv.className = 'alert alert-danger';
                alertDiv.innerHTML = `<i class="fas fa-exclamation-triangle me-2"></i>${err.message}`;
                form.prepend(alertDiv);

                submitBtn.disabled = false;
                submitBtn.innerHTML = originalText;
            }
        });
    </script>
    </body>
    </html>
    <?php
}

/**
 * Generate HTML for a single form field based on its config
 */
function generateFieldHTML($field) {
    $required = !empty($field['required']) ? 'required' : '';
    $label = htmlspecialchars($field['label'] ?? '');
    $name = htmlspecialchars($field['name'] ?? '');
    $placeholder = htmlspecialchars($field['placeholder'] ?? '');
    $id = htmlspecialchars($field['id'] ?? uniqid('field_'));

    $html = "<label for=\"{$id}\" class=\"form-label\">{$label}";
    if ($required) {
        $html .= ' <span class="text-danger">*</span>';
    }
    $html .= "</label>";

    switch ($field['type'] ?? 'text') {
        case 'text':
        case 'email':
        case 'tel':
        case 'number':
        case 'date':
            $html .= "<input type=\"{$field['type']}\" class=\"form-control\" id=\"{$id}\" name=\"{$name}\" placeholder=\"{$placeholder}\" {$required}>";
            break;

        case 'textarea':
            $html .= "<textarea class=\"form-control\" id=\"{$id}\" name=\"{$name}\" rows=\"4\" placeholder=\"{$placeholder}\" {$required}></textarea>";
            break;

        case 'select':
            $html .= "<select class=\"form-select\" id=\"{$id}\" name=\"{$name}\" {$required}>";
            $html .= "<option value=\"\">Choose...</option>";
            foreach ($field['options'] ?? [] as $option) {
                $opt = htmlspecialchars($option);
                $html .= "<option value=\"{$opt}\">{$opt}</option>";
            }
            $html .= "</select>";
            break;

        case 'checkbox':
            $html = "<div class=\"form-check\">";
            $html .= "<input class=\"form-check-input\" type=\"checkbox\" id=\"{$id}\" name=\"{$name}\" value=\"1\" {$required}>";
            $html .= "<label class=\"form-check-label\" for=\"{$id}\">{$label}";
            if ($required) $html .= ' <span class="text-danger">*</span>';
            $html .= "</label></div>";
            break;

        case 'radio':
            $html = "<fieldset><legend class=\"form-label\">{$label}";
            if ($required) $html .= ' <span class="text-danger">*</span>';
            $html .= "</legend>";
            foreach ($field['options'] ?? [] as $idx => $option) {
                $opt = htmlspecialchars($option);
                $optId = $id . '_' . $idx;
                $html .= "<div class=\"form-check\">";
                $html .= "<input class=\"form-check-input\" type=\"radio\" id=\"{$optId}\" name=\"{$name}\" value=\"{$opt}\" {$required}>";
                $html .= "<label class=\"form-check-label\" for=\"{$optId}\">{$opt}</label>";
                $html .= "</div>";
            }
            $html .= "</fieldset>";
            break;

        case 'file':
            $html .= "<input type=\"file\" class=\"form-control\" id=\"{$id}\" name=\"{$name}\" {$required}>";
            break;

        default:
            // Default to text input
            $html .= "<input type=\"text\" class=\"form-control\" id=\"{$id}\" name=\"{$name}\" placeholder=\"{$placeholder}\" {$required}>";
            break;
    }

    return $html;
}

/**
 * Show thank you page after successful submission
 */
function showSuccessPage($formName) {
    ?>
    <!DOCTYPE html>
    <html lang="en">
    <head>
        <meta charset="UTF-8" />
        <meta name="viewport" content="width=device-width, initial-scale=1" />
        <title>Thank You</title>
        <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet" />
        <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet" />
        <style>
            body { background: #f8fafc; }
            .success-container { max-width: 500px; margin: 4rem auto; text-align: center; }
        </style>
    </head>
    <body>
    <div class="container">
        <div class="success-container">
            <div class="card">
                <div class="card-body py-5">
                    <i class="fas fa-check-circle text-success fa-4x mb-4"></i>
                    <h2 class="mb-3">Thank You!</h2>
                    <p class="text-muted">Your form submission has been received successfully.</p>
                    <small class="text-muted">Form: <?php echo htmlspecialchars($formName); ?></small>
                </div>
            </div>
        </div>
    </div>
    </body>
    </html>
    <?php
}

/**
 * Show error page on submission failure
 */
function showErrorPage() {
    ?>
    <!DOCTYPE html>
    <html lang="en">
    <head>
        <meta charset="UTF-8" />
        <meta name="viewport" content="width=device-width, initial-scale=1" />
        <title>Error</title>
        <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet" />
        <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet" />
        <style>
            body { background: #f8fafc; }
            .error-container { max-width: 500px; margin: 4rem auto; text-align: center; }
        </style>
    </head>
    <body>
    <div class="container">
        <div class="error-container">
            <div class="card">
                <div class="card-body py-5">
                    <i class="fas fa-exclamation-triangle text-danger fa-4x mb-4"></i>
                    <h2 class="mb-3">Oops!</h2>
                    <p class="text-muted">There was an error processing your submission. Please try again.</p>
                    <button onclick="history.back()" class="btn btn-primary mt-3">
                        <i class="fas fa-arrow-left me-2"></i>Go Back
                    </button>
                </div>
            </div>
        </div>
    </div>
    </body>
    </html>
    <?php
}