<?php
session_start();

// Define root path to app folder for includes
define('APP_PATH', realpath(__DIR__ . '/../app'));

// Autoload or require needed classes/configs
require_once APP_PATH . '/config/config.php';
require_once APP_PATH . '/config/database.php';
require_once APP_PATH . '/models/user.php';
require_once APP_PATH . '/models/form.php';

// Redirect to login if not logged in
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit();
}

try {
    $database = new Database();
    $db = $database->getConnection();

    $user = new User($db);
    $form = new Form($db);

    // Load user data
    if (!$user->getById($_SESSION['user_id'])) {
        // User not found, logout
        session_destroy();
        header('Location: login.php');
        exit();
    }

    // Get user forms
    $user_forms = $form->getByUserId($_SESSION['user_id']);

    // Get current plan info
    $current_plan = null;
    if (!empty($user->plan_id)) {
        $plan_query = "SELECT * FROM plans WHERE id = :plan_id LIMIT 1";
        $plan_stmt = $db->prepare($plan_query);
        $plan_stmt->bindParam(':plan_id', $user->plan_id, PDO::PARAM_INT);
        $plan_stmt->execute();
        $current_plan = $plan_stmt->fetch(PDO::FETCH_ASSOC);
    }

    // Get submission statistics
    $stats_query = "
        SELECT
            COUNT(s.id) AS total_submissions,
            COUNT(CASE WHEN DATE(s.created_at) = CURDATE() THEN 1 END) AS today_submissions,
            COUNT(CASE WHEN MONTH(s.created_at) = MONTH(CURDATE()) AND YEAR(s.created_at) = YEAR(CURDATE()) THEN 1 END) AS monthly_submissions
        FROM submissions s
        JOIN forms f ON s.form_id = f.id
        WHERE f.user_id = :user_id
    ";
    $stats_stmt = $db->prepare($stats_query);
    $stats_stmt->bindParam(':user_id', $_SESSION['user_id'], PDO::PARAM_INT);
    $stats_stmt->execute();
    $stats = $stats_stmt->fetch(PDO::FETCH_ASSOC);

} catch (PDOException $e) {
    // Handle DB error gracefully
    die("Database error: " . htmlspecialchars($e->getMessage()));
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8" />
    <meta name="viewport" content="width=device-width, initial-scale=1" />
    <title>Dashboard - SmartForm</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet" />
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet" />
    <link href="https://cdn.jsdelivr.net/npm/chart.js@3.9.1/dist/chart.min.css" rel="stylesheet" />
    <link href="assets/css/style.css" rel="stylesheet" />
</head>
<body>
<!-- Navigation -->
<nav class="navbar navbar-expand-lg navbar-dark bg-primary">
    <div class="container-fluid">
        <a class="navbar-brand fw-bold" href="index.php"><i class="fas fa-bolt"></i> SmartForm</a>
        <div class="navbar-nav ms-auto">
            <div class="nav-item dropdown">
                <a class="nav-link dropdown-toggle" href="#" id="navbarDropdown" role="button" data-bs-toggle="dropdown">
                    <i class="fas fa-user-circle me-1"></i> <?php echo htmlspecialchars($user->username); ?>
                </a>
                <ul class="dropdown-menu" aria-labelledby="navbarDropdown">
                    <li><a class="dropdown-item" href="profile.php"><i class="fas fa-user me-2"></i>Profile</a></li>
                    <li><a class="dropdown-item" href="billing.php"><i class="fas fa-credit-card me-2"></i>Billing</a></li>
                    <li><hr class="dropdown-divider" /></li>
                    <li><a class="dropdown-item" href="logout.php"><i class="fas fa-sign-out-alt me-2"></i>Logout</a></li>
                </ul>
            </div>
        </div>
    </div>
</nav>

<div class="container-fluid">
    <div class="row">
        <!-- Sidebar -->
        <nav class="col-md-3 col-lg-2 dashboard-sidebar p-0 bg-light">
            <div class="p-3">
                <ul class="nav flex-column dashboard-nav">
                    <li class="nav-item">
                        <a class="nav-link active" href="dashboard.php"><i class="fas fa-tachometer-alt me-2"></i>Dashboard</a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link" href="forms.php"><i class="fas fa-wpforms me-2"></i>Forms</a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link" href="submissions.php"><i class="fas fa-paper-plane me-2"></i>Submissions</a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link" href="analytics.php"><i class="fas fa-chart-bar me-2"></i>Analytics</a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link" href="integrations.php"><i class="fas fa-plug me-2"></i>Integrations</a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link" href="settings.php"><i class="fas fa-cog me-2"></i>Settings</a>
                    </li>
                </ul>
            </div>
        </nav>

        <!-- Main Content -->
        <main class="col-md-9 col-lg-10">
            <div class="p-4">
                <!-- Header -->
                <div class="d-flex justify-content-between align-items-center mb-4">
                    <div>
                        <h1 class="h3 mb-0">Dashboard</h1>
                        <p class="text-muted">Welcome back, <?php echo htmlspecialchars($user->first_name); ?>!</p>
                    </div>
                    <div>
                        <a href="forms.php?action=create" class="btn btn-primary">
                            <i class="fas fa-plus me-2"></i>New Form
                        </a>
                    </div>
                </div>

                <!-- Plan Status -->
                <?php if ($current_plan): ?>
                <div class="alert alert-info d-flex align-items-center mb-4">
                    <i class="fas fa-info-circle me-3"></i>
                    <div class="flex-grow-1">
                        <strong>Current Plan: <?php echo htmlspecialchars($current_plan['name']); ?></strong>
                        <div class="small">
                            <?php if ($current_plan['submissions_limit'] == -1): ?>
                                Unlimited submissions
                            <?php else: ?>
                                <?php echo (int)$stats['monthly_submissions']; ?> / <?php echo (int)$current_plan['submissions_limit']; ?> submissions this month
                            <?php endif; ?>
                        </div>
                    </div>
                    <?php if ($current_plan['id'] != 4): ?>
                        <a href="billing.php" class="btn btn-outline-primary btn-sm">Upgrade</a>
                    <?php endif; ?>
                </div>
                <?php endif; ?>

                <!-- Statistics Cards -->
                <div class="row stats-container mb-4">
                    <div class="col-md-3">
                        <div class="stats-card">
                            <div class="d-flex align-items-center">
                                <div class="stats-icon bg-primary text-white me-3">
                                    <i class="fas fa-wpforms"></i>
                                </div>
                                <div>
                                    <h3 class="mb-0"><?php echo count($user_forms); ?></h3>
                                    <p class="text-muted mb-0">Active Forms</p>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="stats-card">
                            <div class="d-flex align-items-center">
                                <div class="stats-icon bg-success text-white me-3">
                                    <i class="fas fa-paper-plane"></i>
                                </div>
                                <div>
                                    <h3 class="mb-0"><?php echo (int)$stats['total_submissions']; ?></h3>
                                    <p class="text-muted mb-0">Total Submissions</p>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="stats-card">
                            <div class="d-flex align-items-center">
                                <div class="stats-icon bg-info text-white me-3">
                                    <i class="fas fa-calendar-day"></i>
                                </div>
                                <div>
                                    <h3 class="mb-0"><?php echo (int)$stats['today_submissions']; ?></h3>
                                    <p class="text-muted mb-0">Today</p>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="stats-card">
                            <div class="d-flex align-items-center">
                                <div class="stats-icon bg-warning text-white me-3">
                                    <i class="fas fa-calendar-alt"></i>
                                </div>
                                <div>
                                    <h3 class="mb-0"><?php echo (int)$stats['monthly_submissions']; ?></h3>
                                    <p class="text-muted mb-0">This Month</p>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <div class="row">
                    <!-- Recent Forms -->
                    <div class="col-lg-8">
                        <div class="card">
                            <div class="card-header d-flex justify-content-between align-items-center">
                                <h5 class="mb-0">Your Forms</h5>
                                <a href="forms.php" class="btn btn-sm btn-outline-primary">View All</a>
                            </div>
                            <div class="card-body">
                                <?php if (empty($user_forms)): ?>
                                    <div class="text-center py-4">
                                        <i class="fas fa-wpforms fa-3x text-muted mb-3"></i>
                                        <h5>No forms yet</h5>
                                        <p class="text-muted">Create your first form to get started</p>
                                        <a href="forms.php?action=create" class="btn btn-primary">
                                            <i class="fas fa-plus me-2"></i>Create Form
                                        </a>
                                    </div>
                                <?php else: ?>
                                    <div class="table-responsive">
                                        <table class="table table-hover">
                                            <thead>
                                            <tr>
                                                <th>Form Name</th>
                                                <th>Submissions</th>
                                                <th>Status</th>
                                                <th>Created</th>
                                                <th>Actions</th>
                                            </tr>
                                            </thead>
                                            <tbody>
                                            <?php foreach (array_slice($user_forms, 0, 5) as $user_form): ?>
                                                <?php
                                                // Get submission count for this form
                                                $sub_query = "SELECT COUNT(*) as count FROM submissions WHERE form_id = :form_id";
                                                $sub_stmt = $db->prepare($sub_query);
                                                $sub_stmt->bindParam(':form_id', $user_form['id'], PDO::PARAM_INT);
                                                $sub_stmt->execute();
                                                $sub_count = $sub_stmt->fetch(PDO::FETCH_ASSOC)['count'];
                                                ?>
                                                <tr>
                                                    <td>
                                                        <strong><?php echo htmlspecialchars($user_form['form_name']); ?></strong><br>
                                                        <small class="text-muted"><?php echo htmlspecialchars(SITE_URL . '/f/' . $user_form['form_endpoint']); ?></small>
                                                    </td>
                                                    <td><?php echo (int)$sub_count; ?></td>
                                                    <td>
                                                        <?php if ($user_form['is_active']): ?>
                                                            <span class="badge bg-success">Active</span>
                                                        <?php else: ?>
                                                            <span class="badge bg-secondary">Inactive</span>
                                                        <?php endif; ?>
                                                    </td>
                                                    <td><?php echo date('M j, Y', strtotime($user_form['created_at'])); ?></td>
                                                    <td>
                                                        <div class="btn-group btn-group-sm">
                                                            <a href="forms.php?action=edit&id=<?php echo (int)$user_form['id']; ?>" class="btn btn-outline-primary" title="Edit Form">
                                                                <i class="fas fa-edit"></i>
                                                            </a>
                                                            <button class="btn btn-outline-secondary btn-copy" data-copy="<?php echo htmlspecialchars(SITE_URL . '/f/' . $user_form['form_endpoint']); ?>" title="Copy URL">
                                                                <i class="fas fa-copy"></i>
                                                            </button>
                                                        </div>
                                                    </td>
                                                </tr>
                                            <?php endforeach; ?>
                                            </tbody>
                                        </table>
                                    </div>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>

                    <!-- Quick Actions -->
                    <div class="col-lg-4">
                        <div class="card">
                            <div class="card-header">
                                <h5 class="mb-0">Quick Actions</h5>
                            </div>
                            <div class="card-body">
                                <div class="d-grid gap-2">
                                    <a href="forms.php?action=create" class="btn btn-primary">
                                        <i class="fas fa-plus me-2"></i>Create New Form
                                    </a>
                                    <a href="submissions.php" class="btn btn-outline-primary">
                                        <i class="fas fa-paper-plane me-2"></i>View Submissions
                                    </a>
                                    <a href="analytics.php" class="btn btn-outline-primary">
                                        <i class="fas fa-chart-bar me-2"></i>View Analytics
                                    </a>
                                    <a href="integrations.php" class="btn btn-outline-primary">
                                        <i class="fas fa-plug me-2"></i>Setup Integrations
                                    </a>
                                </div>
                            </div>
                        </div>

                        <!-- Recent Activity -->
                        <div class="card mt-4">
                            <div class="card-header">
                                <h5 class="mb-0">Recent Activity</h5>
                            </div>
                            <div class="card-body">
                                <?php
                                // Get recent submissions
                                $activity_query = "
                                    SELECT s.*, f.form_name 
                                    FROM submissions s 
                                    JOIN forms f ON s.form_id = f.id 
                                    WHERE f.user_id = :user_id 
                                    ORDER BY s.created_at DESC 
                                    LIMIT 5
                                ";
                                $activity_stmt = $db->prepare($activity_query);
                                $activity_stmt->bindParam(':user_id', $_SESSION['user_id'], PDO::PARAM_INT);
                                $activity_stmt->execute();
                                $recent_activity = $activity_stmt->fetchAll(PDO::FETCH_ASSOC);
                                ?>

                                <?php if (empty($recent_activity)): ?>
                                    <p class="text-muted text-center">No recent activity</p>
                                <?php else: ?>
                                    <div class="list-group list-group-flush">
                                        <?php foreach ($recent_activity as $activity): ?>
                                            <div class="list-group-item px-0">
                                                <div class="d-flex align-items-center">
                                                    <div class="flex-shrink-0">
                                                        <i class="fas fa-paper-plane text-primary"></i>
                                                    </div>
                                                    <div class="flex-grow-1 ms-3">
                                                        <p class="mb-1 small">New submission</p>
                                                        <p class="mb-1 fw-bold small"><?php echo htmlspecialchars($activity['form_name']); ?></p>
                                                        <p class="mb-0 text-muted small"><?php echo date('M j, g:i A', strtotime($activity['created_at'])); ?></p>
                                                    </div>
                                                </div>
                                            </div>
                                        <?php endforeach; ?>
                                    </div>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </main>
    </div>
</div>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/chart.js@3.9.1/dist/chart.min.js"></script>
<script src="assets/js/main.js"></script>

<script>
// Copy form URL to clipboard
document.querySelectorAll('.btn-copy').forEach(button => {
    button.addEventListener('click', () => {
        const url = button.getAttribute('data-copy');
        navigator.clipboard.writeText(url).then(() => {
            // Optional: Notify user of success
            alert('Form URL copied to clipboard!');
        }).catch(err => {
            alert('Failed to copy: ' + err);
        });
    });
});
</script>

</body>
</html>