<?php
session_start();
require_once '../../app/Config/Config.php';
require_once '../../app/Config/Database.php';
require_once '../../app/Models/Form.php';

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    http_response_code(401);
    echo json_encode(['success' => false, 'message' => 'Unauthorized']);
    exit();
}

$database = new Database();
$db = $database->getConnection();
$form = new Form($db);

$method = $_SERVER['REQUEST_METHOD'];

switch ($method) {
    case 'GET':
        handleGet();
        break;
    case 'POST':
        handlePost();
        break;
    case 'PUT':
        handlePut();
        break;
    case 'DELETE':
        handleDelete();
        break;
    default:
        http_response_code(405);
        echo json_encode(['success' => false, 'message' => 'Method not allowed']);
        break;
}

function handleGet() {
    global $form;
    
    $form_id = $_GET['id'] ?? '';
    
    try {
        if ($form_id) {
            $form_data = $form->getById($form_id);
            if ($form_data && $form_data['user_id'] == $_SESSION['user_id']) {
                echo json_encode(['success' => true, 'data' => $form_data]);
            } else {
                http_response_code(404);
                echo json_encode(['success' => false, 'message' => 'Form not found']);
            }
        } else {
            $forms = $form->getByUserId($_SESSION['user_id']);
            echo json_encode(['success' => true, 'data' => $forms]);
        }
    } catch (Exception $e) {
        http_response_code(500);
        echo json_encode(['success' => false, 'message' => 'Failed to fetch forms']);
    }
}

function handlePost() {
    global $form;
    
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (empty($input['form_name'])) {
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => 'Form name is required']);
        return;
    }
    
    try {
        $form->user_id = $_SESSION['user_id'];
        $form->form_name = $input['form_name'];
        $form->form_endpoint = $form->generateUniqueEndpoint();
        $form->form_config = json_encode($input['config'] ?? []);
        $form->is_active = $input['is_active'] ?? true;
        
        if ($form->create()) {
            echo json_encode([
                'success' => true,
                'message' => 'Form created successfully',
                'data' => ['id' => $form->id, 'endpoint' => $form->form_endpoint]
            ]);
        } else {
            throw new Exception('Failed to create form');
        }
    } catch (Exception $e) {
        http_response_code(500);
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

function handlePut() {
    global $form;
    
    $input = json_decode(file_get_contents('php://input'), true);
    $form_id = $input['id'] ?? '';
    
    if (empty($form_id)) {
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => 'Form ID is required']);
        return;
    }
    
    try {
        // Verify ownership
        $existing_form = $form->getById($form_id);
        if (!$existing_form || $existing_form['user_id'] != $_SESSION['user_id']) {
            http_response_code(404);
            echo json_encode(['success' => false, 'message' => 'Form not found']);
            return;
        }
        
        $form->id = $form_id;
        $form->form_name = $input['form_name'] ?? $existing_form['form_name'];
        $form->form_config = json_encode($input['config'] ?? json_decode($existing_form['form_config'], true));
        $form->is_active = $input['is_active'] ?? $existing_form['is_active'];
        
        if ($form->update()) {
            echo json_encode(['success' => true, 'message' => 'Form updated successfully']);
        } else {
            throw new Exception('Failed to update form');
        }
    } catch (Exception $e) {
        http_response_code(500);
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

function handleDelete() {
    global $form, $db;
    
    $input = json_decode(file_get_contents('php://input'), true);
    $form_id = $input['id'] ?? '';
    
    if (empty($form_id)) {
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => 'Form ID is required']);
        return;
    }
    
    try {
        // Verify ownership
        $existing_form = $form->getById($form_id);
        if (!$existing_form || $existing_form['user_id'] != $_SESSION['user_id']) {
            http_response_code(404);
            echo json_encode(['success' => false, 'message' => 'Form not found']);
            return;
        }
        
        $form->id = $form_id;
        if ($form->delete()) {
            echo json_encode(['success' => true, 'message' => 'Form deleted successfully']);
        } else {
            throw new Exception('Failed to delete form');
        }
    } catch (Exception $e) {
        http_response_code(500);
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}
?>