<?php

namespace SmartForm\Models;

use SmartForm\Config\Database;
use PDO;
use PDOException;

class Submission
{
    private $db;
    private $table = 'submissions';
    
    // Properties
    public $id;
    public $form_id;
    public $submission_data;
    public $ip_address;
    public $user_agent;
    public $referrer;
    public $spam_score;
    public $is_spam;
    public $is_read;
    public $created_at;
    
    public function __construct()
    {
        $database = Database::getInstance();
        $this->db = $database->getConnection();
    }
    
    /**
     * Create a new submission
     */
    public function create(array $data): int
    {
        try {
            $sql = "INSERT INTO {$this->table} (
                form_id, 
                submission_data, 
                ip_address, 
                user_agent, 
                referrer, 
                spam_score, 
                is_spam,
                created_at
            ) VALUES (?, ?, ?, ?, ?, ?, ?, NOW())";
            
            $stmt = $this->db->prepare($sql);
            $stmt->execute([
                $data['form_id'],
                json_encode($data['submission_data']),
                $data['ip_address'] ?? null,
                $data['user_agent'] ?? null,
                $data['referrer'] ?? null,
                $data['spam_score'] ?? 0.0,
                $data['is_spam'] ?? false
            ]);
            
            $this->id = (int) $this->db->lastInsertId();
            return $this->id;
            
        } catch (PDOException $e) {
            error_log("Error creating submission: " . $e->getMessage());
            throw new \Exception("Failed to create submission");
        }
    }
    
    /**
     * Find submission by ID
     */
    public function find(int $id): ?array
    {
        try {
            $sql = "SELECT s.*, f.form_name, f.user_id 
                    FROM {$this->table} s 
                    LEFT JOIN forms f ON s.form_id = f.id 
                    WHERE s.id = ?";
            
            $stmt = $this->db->prepare($sql);
            $stmt->execute([$id]);
            
            $result = $stmt->fetch();
            
            if ($result) {
                $result['submission_data'] = json_decode($result['submission_data'], true);
                return $result;
            }
            
            return null;
            
        } catch (PDOException $e) {
            error_log("Error finding submission: " . $e->getMessage());
            return null;
        }
    }
    
    /**
     * Get submissions by form ID
     */
    public function getByFormId(int $formId, array $options = []): array
    {
        try {
            $limit = $options['limit'] ?? 20;
            $offset = $options['offset'] ?? 0;
            $orderBy = $options['order_by'] ?? 'created_at';
            $orderDir = $options['order_dir'] ?? 'DESC';
            $includeSpam = $options['include_spam'] ?? false;
            
            $sql = "SELECT s.*, f.form_name 
                    FROM {$this->table} s 
                    LEFT JOIN forms f ON s.form_id = f.id 
                    WHERE s.form_id = ?";
            
            $params = [$formId];
            
            if (!$includeSpam) {
                $sql .= " AND s.is_spam = 0";
            }
            
            $sql .= " ORDER BY s.{$orderBy} {$orderDir} LIMIT ? OFFSET ?";
            $params[] = $limit;
            $params[] = $offset;
            
            $stmt = $this->db->prepare($sql);
            $stmt->execute($params);
            
            $results = $stmt->fetchAll();
            
            // Decode JSON data
            foreach ($results as &$result) {
                $result['submission_data'] = json_decode($result['submission_data'], true);
            }
            
            return $results;
            
        } catch (PDOException $e) {
            error_log("Error getting submissions by form ID: " . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Get submissions by user ID
     */
    public function getByUserId(int $userId, array $options = []): array
    {
        try {
            $limit = $options['limit'] ?? 20;
            $offset = $options['offset'] ?? 0;
            $dateFrom = $options['date_from'] ?? null;
            $dateTo = $options['date_to'] ?? null;
            $search = $options['search'] ?? null;
            $formId = $options['form_id'] ?? null;
            
            $sql = "SELECT s.*, f.form_name 
                    FROM {$this->table} s 
                    INNER JOIN forms f ON s.form_id = f.id 
                    WHERE f.user_id = ?";
            
            $params = [$userId];
            
            if ($formId) {
                $sql .= " AND s.form_id = ?";
                $params[] = $formId;
            }
            
            if ($dateFrom) {
                $sql .= " AND s.created_at >= ?";
                $params[] = $dateFrom;
            }
            
            if ($dateTo) {
                $sql .= " AND s.created_at <= ?";
                $params[] = $dateTo;
            }
            
            if ($search) {
                $sql .= " AND s.submission_data LIKE ?";
                $params[] = "%{$search}%";
            }
            
            $sql .= " ORDER BY s.created_at DESC LIMIT ? OFFSET ?";
            $params[] = $limit;
            $params[] = $offset;
            
            $stmt = $this->db->prepare($sql);
            $stmt->execute($params);
            
            $results = $stmt->fetchAll();
            
            // Decode JSON data
            foreach ($results as &$result) {
                $result['submission_data'] = json_decode($result['submission_data'], true);
            }
            
            return $results;
            
        } catch (PDOException $e) {
            error_log("Error getting submissions by user ID: " . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Count submissions
     */
    public function count(array $filters = []): int
    {
        try {
            $sql = "SELECT COUNT(*) as total FROM {$this->table} s";
            $params = [];
            $conditions = [];
            
            if (isset($filters['form_id'])) {
                $conditions[] = "s.form_id = ?";
                $params[] = $filters['form_id'];
            }
            
            if (isset($filters['user_id'])) {
                $sql .= " INNER JOIN forms f ON s.form_id = f.id";
                $conditions[] = "f.user_id = ?";
                $params[] = $filters['user_id'];
            }
            
            if (isset($filters['is_spam'])) {
                $conditions[] = "s.is_spam = ?";
                $params[] = $filters['is_spam'];
            }
            
            if (isset($filters['date_from'])) {
                $conditions[] = "s.created_at >= ?";
                $params[] = $filters['date_from'];
            }
            
            if (isset($filters['date_to'])) {
                $conditions[] = "s.created_at <= ?";
                $params[] = $filters['date_to'];
            }
            
            if (!empty($conditions)) {
                $sql .= " WHERE " . implode(" AND ", $conditions);
            }
            
            $stmt = $this->db->prepare($sql);
            $stmt->execute($params);
            
            $result = $stmt->fetch();
            return (int) $result['total'];
            
        } catch (PDOException $e) {
            error_log("Error counting submissions: " . $e->getMessage());
            return 0;
        }
    }
    
    /**
     * Mark submission as read
     */
    public function markAsRead(int $id): bool
    {
        try {
            $sql = "UPDATE {$this->table} SET is_read = 1 WHERE id = ?";
            $stmt = $this->db->prepare($sql);
            return $stmt->execute([$id]);
            
        } catch (PDOException $e) {
            error_log("Error marking submission as read: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Mark submission as spam
     */
    public function markAsSpam(int $id): bool
    {
        try {
            $sql = "UPDATE {$this->table} SET is_spam = 1 WHERE id = ?";
            $stmt = $this->db->prepare($sql);
            return $stmt->execute([$id]);
            
        } catch (PDOException $e) {
            error_log("Error marking submission as spam: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Delete submission
     */
    public function delete(int $id): bool
    {
        try {
            $sql = "DELETE FROM {$this->table} WHERE id = ?";
            $stmt = $this->db->prepare($sql);
            return $stmt->execute([$id]);
            
        } catch (PDOException $e) {
            error_log("Error deleting submission: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Get recent submissions
     */
    public function getRecent(int $limit = 10): array
    {
        try {
            $sql = "SELECT s.*, f.form_name 
                    FROM {$this->table} s 
                    LEFT JOIN forms f ON s.form_id = f.id 
                    WHERE s.is_spam = 0 
                    ORDER BY s.created_at DESC 
                    LIMIT ?";
            
            $stmt = $this->db->prepare($sql);
            $stmt->execute([$limit]);
            
            $results = $stmt->fetchAll();
            
            // Decode JSON data
            foreach ($results as &$result) {
                $result['submission_data'] = json_decode($result['submission_data'], true);
            }
            
            return $results;
            
        } catch (PDOException $e) {
            error_log("Error getting recent submissions: " . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Get submissions for export
     */
    public function getForExport(int $formId, array $options = []): array
    {
        try {
            $dateFrom = $options['date_from'] ?? null;
            $dateTo = $options['date_to'] ?? null;
            $includeSpam = $options['include_spam'] ?? false;
            
            $sql = "SELECT s.submission_data, s.ip_address, s.created_at, s.is_spam
                    FROM {$this->table} s 
                    WHERE s.form_id = ?";
            
            $params = [$formId];
            
            if (!$includeSpam) {
                $sql .= " AND s.is_spam = 0";
            }
            
            if ($dateFrom) {
                $sql .= " AND s.created_at >= ?";
                $params[] = $dateFrom;
            }
            
            if ($dateTo) {
                $sql .= " AND s.created_at <= ?";
                $params[] = $dateTo;
            }
            
            $sql .= " ORDER BY s.created_at ASC";
            
            $stmt = $this->db->prepare($sql);
            $stmt->execute($params);
            
            $results = $stmt->fetchAll();
            
            // Decode JSON data and flatten structure
            $exportData = [];
            foreach ($results as $result) {
                $submissionData = json_decode($result['submission_data'], true);
                $submissionData['_ip_address'] = $result['ip_address'];
                $submissionData['_submitted_at'] = $result['created_at'];
                $submissionData['_is_spam'] = $result['is_spam'];
                
                $exportData[] = $submissionData;
            }
            
            return $exportData;
            
        } catch (PDOException $e) {
            error_log("Error getting submissions for export: " . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Get submission statistics
     */
    public function getStats(int $formId, array $options = []): array
    {
        try {
            $dateFrom = $options['date_from'] ?? date('Y-m-d', strtotime('-30 days'));
            $dateTo = $options['date_to'] ?? date('Y-m-d');
            
            $sql = "SELECT 
                        COUNT(*) as total_submissions,
                        COUNT(CASE WHEN is_spam = 0 THEN 1 END) as valid_submissions,
                        COUNT(CASE WHEN is_spam = 1 THEN 1 END) as spam_submissions,
                        COUNT(CASE WHEN is_read = 0 AND is_spam = 0 THEN 1 END) as unread_submissions,
                        AVG(spam_score) as avg_spam_score
                    FROM {$this->table} 
                    WHERE form_id = ? 
                    AND created_at BETWEEN ? AND ?";
            
            $stmt = $this->db->prepare($sql);
            $stmt->execute([$formId, $dateFrom, $dateTo]);
            
            $stats = $stmt->fetch();
            
            // Get daily breakdown
            $dailySQL = "SELECT 
                            DATE(created_at) as date,
                            COUNT(*) as submissions,
                            COUNT(CASE WHEN is_spam = 0 THEN 1 END) as valid_submissions
                         FROM {$this->table} 
                         WHERE form_id = ? 
                         AND created_at BETWEEN ? AND ?
                         GROUP BY DATE(created_at)
                         ORDER BY date ASC";
            
            $dailyStmt = $this->db->prepare($dailySQL);
            $dailyStmt->execute([$formId, $dateFrom, $dateTo]);
            
            $stats['daily_breakdown'] = $dailyStmt->fetchAll();
            
            return $stats;
            
        } catch (PDOException $e) {
            error_log("Error getting submission stats: " . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Validate submission data
     */
    public function validateData(array $data, array $formFields): array
    {
        $errors = [];
        $validatedData = [];
        
        foreach ($formFields as $field) {
            $fieldId = $field['id'];
            $fieldType = $field['type'];
            $isRequired = $field['required'] ?? false;
            $value = $data[$fieldId] ?? null;
            
            // Check required fields
            if ($isRequired && empty($value)) {
                $errors[$fieldId] = "This field is required";
                continue;
            }
            
            // Skip validation if field is empty and not required
            if (empty($value) && !$isRequired) {
                $validatedData[$fieldId] = '';
                continue;
            }
            
            // Type-specific validation
            switch ($fieldType) {
                case 'email':
                    if (!filter_var($value, FILTER_VALIDATE_EMAIL)) {
                        $errors[$fieldId] = "Please enter a valid email address";
                    } else {
                        $validatedData[$fieldId] = filter_var($value, FILTER_SANITIZE_EMAIL);
                    }
                    break;
                    
                case 'phone':
                    $cleanPhone = preg_replace('/[^0-9+\-\(\)\s]/', '', $value);
                    if (strlen($cleanPhone) < 10) {
                        $errors[$fieldId] = "Please enter a valid phone number";
                    } else {
                        $validatedData[$fieldId] = $cleanPhone;
                    }
                    break;
                    
                case 'number':
                    if (!is_numeric($value)) {
                        $errors[$fieldId] = "Please enter a valid number";
                    } else {
                        $validatedData[$fieldId] = (float) $value;
                    }
                    break;
                    
                case 'url':
                    if (!filter_var($value, FILTER_VALIDATE_URL)) {
                        $errors[$fieldId] = "Please enter a valid URL";
                    } else {
                        $validatedData[$fieldId] = filter_var($value, FILTER_SANITIZE_URL);
                    }
                    break;
                    
                case 'date':
                    $date = \DateTime::createFromFormat('Y-m-d', $value);
                    if (!$date || $date->format('Y-m-d') !== $value) {
                        $errors[$fieldId] = "Please enter a valid date";
                    } else {
                        $validatedData[$fieldId] = $value;
                    }
                    break;
                    
                default:
                    // Text, textarea, select, radio, checkbox
                    $validatedData[$fieldId] = htmlspecialchars(trim($value), ENT_QUOTES, 'UTF-8');
                    
                    // Check length constraints
                    if (isset($field['max_length']) && strlen($validatedData[$fieldId]) > $field['max_length']) {
                        $errors[$fieldId] = "This field cannot exceed {$field['max_length']} characters";
                    }
                    
                    if (isset($field['min_length']) && strlen($validatedData[$fieldId]) < $field['min_length']) {
                        $errors[$fieldId] = "This field must be at least {$field['min_length']} characters";
                    }
                    break;
            }
        }
        
        return [
            'valid' => empty($errors),
            'errors' => $errors,
            'data' => $validatedData
        ];
    }
    
    /**
     * Convert to array
     */
    public function toArray(): array
    {
        return [
            'id' => $this->id,
            'form_id' => $this->form_id,
            'submission_data' => is_string($this->submission_data) 
                ? json_decode($this->submission_data, true) 
                : $this->submission_data,
            'ip_address' => $this->ip_address,
            'user_agent' => $this->user_agent,
            'referrer' => $this->referrer,
            'spam_score' => $this->spam_score,
            'is_spam' => $this->is_spam,
            'is_read' => $this->is_read,
            'created_at' => $this->created_at
        ];
    }
}